"use strict";

const Logger 	= require("berry-frame/utils/Logger.js").Logger;
const isMissing = require("berry-frame/utils/Logger.js").isMissing;
const isPresent = require("berry-frame/utils/Logger.js").isPresent;
const App 		= require("berry-frame/server/App.js").App;

var app;

// =========================================================================================================

class Cuckoo extends App {
	/*
		Cuckoo coordinates the cuckoo clock.
	*/
	
	constructor(hardware,silent) {
		super(hardware,silent);
		
		// make instance available for functions which are invoked as callbacks
		app=this;
		
		this.birds = [
			"birds/1/Amsel.mp3",
			"birds/1/Blaumeise.mp3",
			"birds/1/Buchfink.mp3",
			"birds/1/Buntspecht.mp3",
			"birds/1/Eichelhaeher.mp3",
			"birds/1/Elster.mp3",
			"birds/1/Feldlerche.mp3",
			"birds/1/Feldsperling.mp3",
			"birds/1/Gartenrotschwanz.mp3",
			"birds/1/Haussperling.mp3",
			"birds/1/Kohlmeise.mp3",
			"birds/1/Maeusebussard.mp3",
			"birds/1/Mehlschwalbe.mp3",
			"birds/1/Nachtigall.mp3",
			"birds/1/Rotkehlchen.mp3",
			"birds/1/Saatkraehe.mp3",
			"birds/1/Singdrossel.mp3",
			"birds/1/Star.mp3",
			"birds/1/Steinkauz.mp3",
			"birds/1/Tuerkentaube.mp3",
			"birds/1/Waldkauz.mp3",
			"birds/1/Waldohreule.mp3",
			"birds/1/Zaunkoenig.mp3",
			"birds/2/Dohle.mp3",
			"birds/2/Fasan.mp3",
			"birds/2/Goldammer.mp3",
			"birds/2/Gruenfink.mp3",
			"birds/2/Habicht.mp3",
			"birds/2/Haubenlerche.mp3",
			"birds/2/Hausrotschwanz.mp3",
			"birds/2/Kiebitz.mp3",
			"birds/2/Kleiber.mp3",
			"birds/2/Neuntoeter.mp3",
			"birds/2/Rauchschwalbe.mp3",
			"birds/2/Rebhuhn.mp3",
			"birds/2/Rotdrossel.mp3",
			"birds/2/Schwarzspecht.mp3",
			"birds/2/Stieglitz.mp3",
			"birds/3/Baumpieper.mp3",
			"birds/3/Bienenfresser.mp3",
			"birds/3/Blauracke.mp3",
			"birds/3/Brachvogel.mp3",
			"birds/3/Dorngrasmuecke.mp3",
			"birds/3/Fitis.mp3",
			"birds/3/Gartenbaumlaeufer.mp3",
			"birds/3/Girlitz.mp3",
			"birds/3/Grauammer.mp3",
			"birds/3/Grauschnaepper.mp3",
			"birds/3/Heckenbraunelle.mp3",
			"birds/3/Klappergrasmuecke.mp3",
			"birds/3/Misteldrossel.mp3",
			"birds/3/Moenchsgrasmuecke.mp3",
			"birds/3/Schwarzkehlchen.mp3",
			"birds/3/Trauerschnaepper.mp3",
			"birds/3/Wacholderdrossel.mp3",
			"birds/3/Wintergoldhaehnchen.mp3",
			"birds/3/Zilpzalp.mp3",
			"birds/4/Braunkehlchen.mp3",
			"birds/4/Gartengrasmuecke.mp3",
			"birds/4/Gelbspoetter.mp3",
			"birds/4/Haenfling.mp3",
			"birds/4/Halsbandschnaepper.mp3",
			"birds/4/Heidelerche.mp3",
			"birds/4/Rabenkraehe.mp3",
			"birds/4/Raubwuerger.mp3",
			"birds/4/Sprosser.mp3",
			"birds/4/Tannenmeise.mp3",
			"birds/4/Triel.mp3",
			"birds/4/Turteltaube.mp3",
			"birds/4/Uferschwalbe.mp3",
			"birds/4/Wachtel.mp3",
			"birds/4/Waldlaubsaenger.mp3",
			"birds/4/Wendehals.mp3",
			"birds/4/Ziegenmelker.mp3",
			"birds/4/Zwergschnaepper.mp3",		
		];
		
	}
	
	init() {
		// called after all hardware elements have been created
		this.strip		= this.hardware.elms.strip.dev;
		this.speakers	= this.hardware.elms.speakers.dev;
		this.motor		= this.hardware.elms.motor.dev;
		this.brightness	= this.hardware.elms.brightness.dev;
	}
	
	onStart(onStarted) {
		// Logger.info("Cuckoo       onStarted - doing nothing so far.");
		if (onStarted) onStarted();
	}
	
	onStop(onFinished) {
		// Logger.info("Cuckoo       onFinished - doing nothing so far.");
		if (onFinished) onFinished();
	}

	setLightAfter(color,delay) {
		setTimeout(function() { this.strip.fillColor(color); }.bind(this), delay);		
	}

	cuckooSongAfter(hour,delay) {
		// let cuckoo sing (12 hours pattern), return duration of song
		var hour12 = (hour%12==0) ? 12 : hour %12;
		setTimeout(function() { this.speakers.play(hour12+".wav"); }.bind(this), delay);
		return 3000+(hour12-1)*800;
	}

	cuckooMoveAfter(direction,delay) {
		// move the cuckoo "out" or "in"
		setTimeout(function() { 
			this.motor.changeDutyCycle(
				direction=="out" ? {value: 0.13, stepSize:0.001,delay: 30} : {value: 0.035, stepSize:0.001,delay: 30}
			);
			}.bind(this), delay);
		return 1500;		// move duration
	}

	birdSongAfter(delay) {
		// let a randomly selected bird sing
		var bird = this.birds[Math.round(this.birds.length*Math.random())];
		Logger.log("Cuckoo       birdSong "+bird);
		setTimeout(function() { this.speakers.play(bird); }.bind(this), delay);
	}

	messageAfter(msg,delay) {
		// play an audio file after delay
		Logger.log("Cuckoo       message "+msg+ " time="+(new Date().toString()));
		setTimeout(function() { 
			// move cuckoo, show light, play message
			this.setLightAfter("#009900",0);									// green light
			var moveDuration = this.cuckooMoveAfter("out",0);					// move Cuckoo out
			var that = this;
			setTimeout(function() { that.speakers.play(msg); }, moveDuration);
			this.cuckooMoveAfter("in",10000+moveDuration);						// move Cuckoo back in
			this.setLightAfter("#000000",10000+2*moveDuration);					// light off
		}.bind(this), delay);
	}
	
	cuckooSay(cb,msg) {
		// play a TTS message announced by cuckoo, ignore callback
		Logger.log("Cuckoo       say "+msg);
		// move cuckoo, show light, play message
		this.setLightAfter("#0099ff",0);									// green light
		var moveDuration = this.cuckooMoveAfter("out",0);					// move Cuckoo out
		var that = this;
		setTimeout(function() { that.speakers.play("announce.wav"); }, moveDuration);
		setTimeout(function() { that.speakers.say(msg); }, moveDuration+3000);
		this.cuckooMoveAfter("in",13000+moveDuration);						// move Cuckoo back in
		this.setLightAfter("#000000",13000+2*moveDuration);					// light off
	}

	onHour() {
		// action on full hour

		// no actions outside 8:00 .. 20:00
		var hour = new Date().getHours();		
		if (hour<8 || hour>20) {
			return;
		}
		
		// no action if it is too dark
		var minBrightness = 50;
		var brightness = this.brightness.getValue();
		if (brightness<minBrightness) {
			Logger.log("Cuckoo       hour "+hour+" , brightness = "+brightness+" -- too dark -- "+(new Date()).toString());
			return;
		}
		
		// start actions
		Logger.log("Cuckoo       hour "+hour+" , brightness = "+brightness);
		var now=0;
		
		// move cuckoo, show light, play cuckoo song
		this.setLightAfter("#ff4400",now);						// orange light
		var moveDuration = this.cuckooMoveAfter("out",now);		// move Cuckoo out
		this.setLightAfter("#bbbb00",now+=moveDuration);		// yellow light
		var cuckooDuration = this.cuckooSongAfter(hour,now);	// Cuckoo song
		this.setLightAfter("#0000ff",now+=cuckooDuration);		// blue light
		this.cuckooMoveAfter("in",now);							// move Cuckoo back in
		this.setLightAfter("#000000",now+=moveDuration);		// light off, Cuckoo sleeping
		
		// at certain occasions play other bird plus explanations
		if ([16,19].includes(hour)) {
//		if (Math.random()<0.5) {		// uncomment if you like as an alternative
			this.birdSongAfter(now+2000);
		}
		
		// message: time to sleep
		if (hour==19) {
			// after 30 minutes
			this.messageAfter("sleep.mp3",30*60*1000);
		}
	}	
	
}
Cuckoo.getApiDescription = function() {
	return [
		{	cmd:	"onHour",
			effect:	"choose a sound and play.",
		},
	];
}
// =========================================================================================================
module.exports.Cuckoo = Cuckoo;
